import { Block } from "@minecraft/server";

export class Particle {

  /**
   * Returns the Bounding Box of a specific block.
   * @param {Block} block The block that should be marked as the base block for the Bounding Box.
   * @param {boolean} accurately If true, the Bounding Box will only be considered up to the block limit. Default: false
   * @param {number} offset Offset to be applied if the `accurately` parameter is marked as false. Offset is applied in all directions of the block. Default: 0.25
   * 
   * @returns {{
   *  minX: number, minY: number, minZ: number,
   *  maxX: number, maxY: number, maxZ: number,
   * }}
   */
  static getBoundingBoxFrom(block, accurately = false, offset = 0.25) {
    const minOffset = accurately ? 0 : offset;
    const maxOffset = accurately ? 1 : 1 + offset;

    return {
      minX: block.location.x - minOffset,
      minY: block.location.y - minOffset,
      minZ: block.location.z - minOffset,
      maxX: block.location.x + maxOffset,
      maxY: block.location.y + maxOffset,
      maxZ: block.location.z + maxOffset,
    }
  }

  /**
   * A random Vector3 inside a Bounding Box.
   * @param {{
   *  minX: number, minY: number, minZ: number,
   *  maxX: number, maxY: number, maxZ: number,
   * }} boundingBox The bounding box. 
   * 
   * @returns {import("@minecraft/server").Vector3} Returns a Vector3.
   */
  static getRandomPositionInsideBoundingBox(boundingBox) {
    return {
      x: Math.random() * (boundingBox.maxX - boundingBox.minX) + boundingBox.minX,
      y: Math.random() * (boundingBox.maxY - boundingBox.minY) + boundingBox.minY,
      z: Math.random() * (boundingBox.maxZ - boundingBox.minZ) + boundingBox.minZ
    }
  }
}